<?php

abstract class rpcModule
{
	/**
	 * @param string $sMethod
	 * @param mixed ...
	 * @return mixed
	 */
	abstract public function doRequest($sMethod);
}

class rpcModuleSoap extends rpcModule
{
	/**
	 * @var string
	 */
	const CONTENT_DE_WSDL = 'http://www.content.de/api/clientservices.php?wsdl';


	/**
	 * @param SoapClient
	 */
	private $oSoapClient = null;

	/**
	 * @return void
	 */
	public function __construct()
	{
		$this->oSoapClient = new SoapClient(self::CONTENT_DE_WSDL);
	}

	/**
	 * @param string $sMethod
	 * @param mixed ...
	 * @return mixed
	 */
	public function doRequest($sMethod)
	{
		if(func_num_args() < 1)
		{
			throw new InvalidArgumentException('not enough arguments');
		}

		$aArguments = func_get_args();

		$sMethod = array_shift($aArguments);

		try
		{
			$mResult = call_user_func_array(array($this->oSoapClient, $sMethod), $aArguments);

			if(is_object($mResult) && isset($mResult->HasError) && $mResult->HasError == true)
			{
				throw new Exception($mResult->ErrorMsg);
			}

			if(is_object($mResult))
			{
				$mResult = $this->normalizeResult($mResult);
			}

			return $mResult;
		}
		catch(SoapFault $oSoapError)
		{
			throw new Exception($oSoapError->getMessage());
		}
	}

	/**
	 * @param stdClass $oResult
	 * @return array
	 */
	private function normalizeResult($oResult)
	{
		 return array(
		 	'HasError' => $oResult->HasError,
		 	'ErrorMsg' => $oResult->ErrorMsg,
		 	'Records' => $oResult->Records,
		 	'NumRecords' => $oResult->NumRecords
		 );
	}
}

class rpcModuleXmlrpc extends rpcModule
{
	/**
	 * @var string
	 */
	const CONTENT_DE_XMLRPC_URL = 'http://www.content.de/api/xmlrpc.php';


	/**
	 * @return void
	 */
	public function __construct()
	{
		if(!extension_loaded('curl'))
		{
			throw new Exception('this module requires cUrl');
		}
	}

	/**
	 * @param string $sMethod
	 * @param mixed ...
	 * @return mixed
	 */
	public function doRequest($sMethod)
	{
		if(func_num_args() < 1)
		{
			throw new InvalidArgumentException('not enough arguments');
		}

		$aArguments = func_get_args();

		$sMethod = $this->getRealMethod(array_shift($aArguments));

		return $this->executeRequest($sMethod, $aArguments);
	}

	/**
	 * @param string $sMethod
	 * @return string
	 */
	private function getRealMethod($sMethod)
	{
		if(strpos($sMethod, 'content.') === false)
		{
			$sMethod = 'content.' . $sMethod;
		}

		return $sMethod;
	}

	/**
	 * @param string $sFunction
	 * @param array $aArguments
	 * @return array
	 */
	private function executeRequest($sMethod, array $aArguments = array())
	{
		$aArguments = $this->encode($aArguments);

		$sRpcRequest = xmlrpc_encode_request($sMethod, $aArguments);

		$rCurl = curl_init(self::CONTENT_DE_XMLRPC_URL);

		$aHeaders = array(
			'content-type: text/xml'
		);

		curl_setopt_array($rCurl, array(
			CURLOPT_POSTFIELDS => $sRpcRequest,
			CURLOPT_FOLLOWLOCATION => true,
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_TIMEOUT => 50,
			CURLOPT_HTTPHEADER => $aHeaders
		));

		$sResponse = curl_exec($rCurl);

		$aResponse = xmlrpc_decode($sResponse);

		if(is_array($aResponse) && isset($aResponse['faultCode']) && isset($aResponse['faultString']))
		{
			throw new Exception($aResponse['faultString'], (int) $aResponse['faultCode']);
		}

		$aResponse = $this->decode($aResponse);

		return $aResponse;
	}

	/**
	 * @param array $aData
	 * @return array
	 */
	private function decode($mData)
	{
		if(is_array($mData))
		{
			foreach($mData as $mIndex => &$mSubData)
			{
				$mSubData = $this->decode($mSubData);
			}

			return $mData;
		}

		if(is_string($mData))
		{

			$mData = html_entity_decode(html_entity_decode(html_entity_decode(htmlentities($mData), ENT_QUOTES, 'utf-8'), ENT_QUOTES, 'utf-8'), ENT_QUOTES, 'utf-8');
		}

		return $mData;
	}

	/**
	 * @param mixed $mData
	 * @return mixed
	 */
	private function encode($mData)
	{
		if(is_array($mData))
		{
			foreach($mData as $mIndex => $mSubData)
			{
				$mData[$mIndex] = $this->encode($mSubData);
			}
		}
		elseif(is_string($mData))
		{
			$mData = htmlentities(utf8_decode($mData), ENT_QUOTES);
		}

		return $mData;
	}
}

?>